<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\State;
use App\Models\District;
use App\Models\City;
use Illuminate\Http\Request;

class LocationController extends Controller
{
    // State APIs
    public function getStates()
    {
        $states = State::whereNull('deleted_at')->where('status', 1)->get();

        return response()->json([
            'success' => true,
            'data' => $states
        ]);
    }

    public function addState(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:states,name'
        ]);

        $state = State::create([
            'name' => $request->name,
            'status' => 1
        ]);

        return response()->json([
            'success' => true,
            'message' => 'State added successfully',
            'data' => $state
        ]);
    }

    public function getState($id)
    {
        $state = State::find($id);

        if (!$state || $state->deleted_at) {
            return response()->json([
                'success' => false,
                'message' => 'State not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $state
        ]);
    }

    public function updateState(Request $request, $id)
    {
        $state = State::find($id);

        if (!$state || $state->deleted_at) {
            return response()->json([
                'success' => false,
                'message' => 'State not found'
            ], 404);
        }

        $request->validate([
            'name' => 'required|string|max:255|unique:states,name,' . $id
        ]);

        $state->update(['name' => $request->name]);

        return response()->json([
            'success' => true,
            'message' => 'State updated successfully',
            'data' => $state
        ]);
    }

    // District APIs
    public function getDistricts($stateId = null)
    {
        $query = District::with('state')->whereNull('deleted_at')->where('status', 1);

        if ($stateId) {
            $query->where('state_id', $stateId);
        }

        $districts = $query->get();

        return response()->json([
            'success' => true,
            'data' => $districts
        ]);
    }

    public function addDistrict(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'state_id' => 'required|exists:states,id'
        ]);

        $district = District::create([
            'name' => $request->name,
            'state_id' => $request->state_id,
            'status' => 1
        ]);

        return response()->json([
            'success' => true,
            'message' => 'District added successfully',
            'data' => $district->load('state')
        ]);
    }

    public function getDistrict($id)
    {
        $district = District::with('state')->find($id);

        if (!$district || $district->deleted_at) {
            return response()->json([
                'success' => false,
                'message' => 'District not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $district
        ]);
    }

    public function updateDistrict(Request $request, $id)
    {
        $district = District::find($id);

        if (!$district || $district->deleted_at) {
            return response()->json([
                'success' => false,
                'message' => 'District not found'
            ], 404);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'state_id' => 'required|exists:states,id'
        ]);

        $district->update([
            'name' => $request->name,
            'state_id' => $request->state_id
        ]);

        return response()->json([
            'success' => true,
            'message' => 'District updated successfully',
            'data' => $district->load('state')
        ]);
    }

    // City APIs
    // public function getCities($districtId = null)
    // {
    //     // dd($districtId);
    //     $query = City::with(['state', 'district'])->whereNull('deleted_at')->where('status', 1);

    //     if ($districtId) {
    //         $query->where('district_id', $districtId);
    //     }

    //     $cities = $query->get();

    //     return response()->json([
    //         'success' => true,
    //         'data' => $cities
    //     ]);
    // }

    public function getCities($stateId = null)
    {
        $query = City::whereNull('deleted_at')
            ->where('status', 1);

        if ($stateId) {
            $query->where('district_id', $stateId);
        }

        return response()->json([
            'success' => true,
            'data' => $query->get()
        ]);
    }


    public function addCity(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'state_id' => 'required|exists:states,id',
            'district_id' => 'required|exists:districts,id'
        ]);

        $city = City::create([
            'name' => $request->name,
            'state_id' => $request->state_id,
            'district_id' => $request->district_id,
            'status' => 1
        ]);

        return response()->json([
            'success' => true,
            'message' => 'City added successfully',
            'data' => $city->load(['state', 'district'])
        ]);
    }

    public function getCity($id)
    {
        $city = City::with(['state', 'district'])->find($id);

        if (!$city || $city->deleted_at) {
            return response()->json([
                'success' => false,
                'message' => 'City not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $city
        ]);
    }

    public function updateCity(Request $request, $id)
    {
        $city = City::find($id);

        if (!$city || $city->deleted_at) {
            return response()->json([
                'success' => false,
                'message' => 'City not found'
            ], 404);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'state_id' => 'required|exists:states,id',
            'district_id' => 'required|exists:districts,id'
        ]);

        $city->update([
            'name' => $request->name,
            'state_id' => $request->state_id,
            'district_id' => $request->district_id
        ]);

        return response()->json([
            'success' => true,
            'message' => 'City updated successfully',
            'data' => $city->load(['state', 'district'])
        ]);
    }
}
