<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Applicant;
use App\Models\FormFieldOption;
use App\Models\State;
use App\Models\City;
use App\Models\User;
use App\Services\ApiLogService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Constants\LeadStatus;

class RoController extends Controller
{
    public function roAssignedLeads(Request $request)
    {
        // Entry log
        ApiLogService::entry('RO assigned leads request received', $request->all());

        try {
            $query = Applicant::with([
                'creator:id,name,mobile,alternate_mobile',
                'assignedTo:id,name',
                'assignedBy:id,name,mobile,alternate_mobile',
                'branch:id,branch_name,branch_code'
            ])->where('assign_to', Auth::id());

            // Filter branches and users based on logged-in user's role
            $currentUser = Auth::user();
            if ($currentUser->role_id != 1) {
                // Non-admin users can only see leads from their own branch
                $query->where('branch_id', $currentUser->branch_id);
            }

            if ($request->filled('search')) {
                if (is_numeric($request->search)) {
                    $query->where('contact_number', 'like', '%' . $request->search . '%');
                } else {
                    $query->where(function ($q) use ($request) {
                        $q->where('applicant_first_name', 'like', '%' . $request->search . '%')
                            ->orWhere('applicant_last_name', 'like', '%' . $request->search . '%');
                    });
                }
            }

            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }

            if ($request->filled('created_by')) {
                $query->where('created_by', $request->created_by);
            }

            if ($request->filled('branch_id')) {
                $query->where('branch_id', $request->branch_id);
            }

            if ($request->filled('bm_name_id')) {
                $query->where('bm_name_id', $request->bm_name_id);
            }

            if ($request->filled('assign_to')) {
                $query->where('assign_to', $request->assign_to);
            }

            if ($request->filled('start_date') && $request->filled('end_date')) {
                $query->whereBetween('created_at', [$request->start_date . ' 00:00:00', $request->end_date . ' 23:59:59']);
            } elseif ($request->filled('start_date')) {
                $query->where('created_at', '>=', $request->start_date . ' 00:00:00');
            } elseif ($request->filled('end_date')) {
                $query->where('created_at', '<=', $request->end_date . ' 23:59:59');
            }

            $applicants = $query->orderBy('created_at', 'desc')->paginate(10);

            // Get form field options for mapping
            $formOptions = FormFieldOption::all()->keyBy('id');
            $states = State::all()->keyBy('id');
            $cities = City::all()->keyBy('id');
            $users = User::all()->keyBy('id');

            // Transform data to replace null with empty strings and exclude unwanted fields
            $applicants->getCollection()->transform(function ($applicant) use ($formOptions, $states, $cities, $users) {
                $applicantArray = $applicant->toArray();

                // Remove unwanted fields
                unset($applicantArray['reference_1']);
                unset($applicantArray['reference_1_relation_id']);
                unset($applicantArray['reference_2']);
                unset($applicantArray['reference_2_relation_id']);
                unset($applicantArray['updated_at']);

                // Add option names for form field IDs
                $fieldMappings = [
                    'segment' => 'segment_name',
                    'loan_type' => 'loan_type_name',
                    'customer_type' => 'customer_type_name',
                    'caste' => 'caste_name',
                    'gender' => 'gender_name',
                    'address_type' => 'address_type_name',
                    'nationality' => 'nationality_name',
                    'marital_status' => 'marital_status_name',
                    'banking_info' => 'banking_info_name',
                    'tenure' => 'tenure_name',
                    'imd_waiver' => 'imd_waiver_name',
                    'property_type' => 'property_type_name',
                    'manufacturing_service_agriculture' => 'manufacturing_service_agriculture_name',
                    'psl_sub_category' => 'psl_sub_category_name',
                    'purpose_of_loans' => 'purpose_of_loans_name',
                    'psl_document_type' => 'psl_document_type_name'
                ];

                foreach ($fieldMappings as $field => $nameField) {
                    if (isset($applicantArray[$field]) && $applicantArray[$field]) {
                        $optionId = $applicantArray[$field];
                        $applicantArray[$nameField] = $formOptions->has($optionId) ? $formOptions[$optionId]->option_value : '';
                    } else {
                        $applicantArray[$nameField] = '';
                    }
                    // Remove the original ID field
                    unset($applicantArray[$field]);
                }

                // Add state, city, and BM names
                if (isset($applicantArray['current_state']) && $applicantArray['current_state']) {
                    $stateId = $applicantArray['current_state'];
                    $applicantArray['current_state_name'] = $states->has($stateId) ? $states[$stateId]->name : '';
                }
                unset($applicantArray['current_state']);

                if (isset($applicantArray['current_city']) && $applicantArray['current_city']) {
                    $cityId = $applicantArray['current_city'];
                    $applicantArray['current_city_name'] = $cities->has($cityId) ? $cities[$cityId]->name : '';
                }
                unset($applicantArray['current_city']);

                if (isset($applicantArray['bm_name_id']) && $applicantArray['bm_name_id']) {
                    $bmId = $applicantArray['bm_name_id'];
                    $applicantArray['bm_name'] = $users->has($bmId) ? $users[$bmId]->name : '';
                }
                unset($applicantArray['bm_name_id']);

                // Replace null values with empty strings recursively
                array_walk_recursive($applicantArray, function (&$value) {
                    if ($value === null) {
                        $value = '';
                    }
                });

                return $applicantArray;
            });

            $response = [
                'status' => true,
                'data' => $applicants,
                'leadStatusLabels' => LeadStatus::map()
            ];

            // Success log
            ApiLogService::success('RO assigned leads retrieved successfully', [
                'total_leads' => $applicants->total(),
                'current_page' => $applicants->currentPage()
            ]);

            // Exit log
            ApiLogService::exit('RO assigned leads request completed successfully', $response);

            return response()->json($response);
        } catch (\Exception $e) {
            // Error log
            Log::error('Error in RoController@roAssignedLeads: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            ApiLogService::error('RO assigned leads request failed with exception', $e);

            return response()->json([
                'status' => false,
                'message' => 'An error occurred while fetching assigned leads.'
            ], 500);
        }
    }
}
