<?php

namespace App\Http\Controllers;

use App\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class DepartmentController extends Controller
{
    public function index()
    {
        try {
            $departments = Department::orderBy('name')->get();
            return view('masters.user-role.department', compact('departments'));
        } catch (\Exception $e) {
            Log::error('Department index error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Unable to load departments.');
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255|unique:departments,name',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            Department::create([
                'name' => $request->name
            ]);

            return redirect()->back()->with('success', 'Department created successfully.');
        } catch (\Exception $e) {
            Log::error('Department store error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Unable to create department.');
        }
    }

    public function update(Request $request, Department $department)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255|unique:departments,name,' . $department->id,
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $department->update([
                'name' => $request->name
            ]);

            return redirect()->back()->with('success', 'Department updated successfully.');
        } catch (\Exception $e) {
            Log::error('Department update error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Unable to update department.');
        }
    }

    public function destroy(Department $department)
    {
        try {
            $department->delete();
            return redirect()->back()->with('success', 'Department deleted successfully.');
        } catch (\Exception $e) {
            Log::error('Department delete error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Unable to delete department.');
        }
    }

    public function toggleStatus(Department $department)
    {
        try {
            $department->update(['status' => !$department->status]);
            $status = $department->status ? 'activated' : 'deactivated';
            return redirect()->back()->with('success', "Department {$status} successfully.");
        } catch (\Exception $e) {
            Log::error('Department toggle status error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Unable to update department status.');
        }
    }
}