<?php

namespace App\Http\Controllers;

use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class DesignationController extends Controller
{
    public function index()
    {
        try {
            $designations = Role::whereNull('deleted_at')->get();
            return view('masters.designation.designation', compact('designations'));
        } catch (\Exception $e) {
            Log::error('Error in DesignationController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching designations.');
        }
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'designation_name' => 'required|string|max:255|unique:roles,name,NULL,id,deleted_at,NULL',
                'authorities' => 'nullable|integer|min:0'
            ]);

            Role::create([
                'name' => $request->designation_name,
                'authorities' => $request->authorities
            ]);

            return redirect()->back()->with('success', 'Designation created successfully');
        } catch (\Exception $e) {
            Log::error('Error in DesignationController@store: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while creating the designation.');
        }
    }

    public function update(Request $request, Role $designation)
    {
        try {
            $request->validate([
                'designation_name' => 'required|string|max:255|unique:roles,name,' . $designation->id . ',id,deleted_at,NULL',
                'authorities' => 'nullable|integer|min:0'
            ]);

            $designation->update([
                'name' => $request->designation_name,
                'authorities' => $request->authorities
            ]);

            return redirect()->back()->with('success', 'Designation updated successfully');
        } catch (\Exception $e) {
            Log::error('Error in DesignationController@update: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating the designation.');
        }
    }

    public function destroy(Role $designation)
    {
        try {
            $designation->delete();
            return redirect()->back()->with('success', 'Designation deleted successfully');
        } catch (\Exception $e) {
            Log::error('Error in DesignationController@destroy: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while deleting the designation.');
        }
    }

    public function toggleStatus(Role $designation)
    {
        try {
            $designation->update(['status' => !$designation->status]);
            $status = $designation->status ? 'activated' : 'deactivated';
            return redirect()->back()->with('success', "Designation {$status} successfully");
        } catch (\Exception $e) {
            Log::error('Error in DesignationController@toggleStatus: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating designation status.');
        }
    }
}
