<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\State;
use App\Models\City;
use App\Models\District;
use App\Models\Role;
use App\Models\Department;
use App\Models\Branch;
use App\Utils\FileUpload;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class EmployeeController extends Controller
{
    public function index(Request $request)
    {
        try {
            $user = Auth::user();
            $userRole = Role::find($user->role_id);

            $query = User::with(['state', 'city', 'role', 'branch'])->whereNull('deleted_at');

            // Filter employees based on user's role and branch
            if ($user->role_id != 1) {
                // Non-admin users can only see employees from their own branch
                $query->where('branch_id', $user->branch_id);
            }

            // Filter employees based on user's authority level
            if ($userRole && $userRole->authorities) {
                $query->whereHas('role', function ($q) use ($userRole) {
                    $q->where('authorities', '>', $userRole->authorities);
                });
            }

            // Apply filters
            if ($request->filled('name')) {
                $query->where('name', 'like', '%' . $request->name . '%');
            }

            if ($request->filled('mobile')) {
                $query->where('mobile', 'like', '%' . $request->mobile . '%');
            }

            if ($request->filled('state_id')) {
                $query->where('state_id', $request->state_id);
            }

            if ($request->filled('role_id')) {
                $query->where('role_id', $request->role_id);
            }

            $employees = $query->get();
            $states = State::whereNull('deleted_at')->get();

            // Filter roles based on user's authority level for filter dropdown
            if ($userRole && $userRole->authorities) {
                $roles = Role::whereNull('deleted_at')
                    ->where('authorities', '>', $userRole->authorities)
                    ->get();
            } else {
                $roles = Role::whereNull('deleted_at')->get();
            }

            return view('team.employee-list', compact('employees', 'states', 'roles'));
        } catch (\Exception $e) {
            Log::error('Error in EmployeeController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching employees.');
        }
    }

    public function create()
    {
        try {
            $user = Auth::user();
            $userRole = Role::find($user->role_id);

            $states = State::whereNull('deleted_at')->get();
            $districts = District::whereNull('deleted_at')->get();
            $cities = City::whereNull('deleted_at')->get();

            // Filter roles based on user's authority level
            if ($userRole && $userRole->authorities) {
                $roles = Role::whereNull('deleted_at')
                    ->where('authorities', '>', $userRole->authorities)
                    ->get();
            } else {
                $roles = Role::whereNull('deleted_at')->get();
            }

            $departments = Department::whereNull('deleted_at')->where('status', 1)->get();
            
            // Filter branches based on user's role
            if ($user->role_id != 1) {
                // Non-admin users can only see their own branch
                $branches = Branch::whereNull('deleted_at')->where('status', 1)->where('id', $user->branch_id)->get();
            } else {
                $branches = Branch::whereNull('deleted_at')->where('status', 1)->get();
            }
            
            return view('team.add-employee', compact('states', 'districts', 'cities', 'roles', 'departments', 'branches'));
        } catch (\Exception $e) {
            Log::error('Error in EmployeeController@create: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the create form.');
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'father_name' => 'nullable|string|max:255',
                'mother_name' => 'nullable|string|max:255',
                'dob' => 'nullable|date',
                'gender' => 'nullable|in:male,female,other',
                'marital_status' => 'nullable|in:single,married,divorced,widowed',
                'email' => 'required|email|unique:users,email',
                'mobile' => 'required|string|max:15|unique:users,mobile',
                'alternate_mobile' => 'nullable|string|max:15',
                'aadhaar_number' => 'nullable|string|size:12',
                'pan_number' => 'nullable|string|size:10',
                'password' => 'required|string|min:6',
                'joining_date' => 'required|date',
                'state_id' => 'required|exists:states,id',
                'district_id' => 'required|exists:districts,id',
                'city_id' => 'required|exists:cities,id',
                'pincode' => 'nullable|string|max:6',
                'address' => 'nullable|string',
                'role_id' => 'required|exists:roles,id',
                'department_id' => 'required|exists:departments,id',
                'branch_id' => 'required|exists:branches,id',
                'profile_pic' => 'nullable|image|mimes:jpeg,png,jpg|max:8048',
                'addhar_front_pic' => 'nullable|image|mimes:jpeg,png,jpg|max:8048',
                'addhar_back_pic' => 'nullable|image|mimes:jpeg,png,jpg|max:8048'
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            // Generate staff ID
            $lastUser = User::whereNotNull('staff_id')
                ->where('staff_id', 'like', 'STF%')
                ->orderByRaw('CAST(SUBSTRING(staff_id, 4) AS UNSIGNED) DESC')
                ->first();

            if ($lastUser && $lastUser->staff_id) {
                $lastNumber = (int)substr($lastUser->staff_id, 3);
                $newNumber = $lastNumber + 1;
            } else {
                $newNumber = 1;
            }

            $staffId = 'STF' . str_pad($newNumber, 3, '0', STR_PAD_LEFT);

            $data = [
                'staff_id' => $staffId,
                'name' => $request->name,
                'father_name' => $request->father_name,
                'mother_name' => $request->mother_name,
                'dob' => $request->dob,
                'gender' => $request->gender,
                'marital_status' => $request->marital_status,
                'email' => $request->email,
                'mobile' => $request->mobile,
                'alternate_mobile' => $request->alternate_mobile,
                'aadhaar_number' => $request->aadhaar_number,
                'pan_number' => $request->pan_number,
                'password' => Hash::make($request->password),
                'joining_date' => $request->joining_date,
                'state_id' => $request->state_id,
                'district_id' => $request->district_id,
                'city_id' => $request->city_id,
                'pincode' => $request->pincode,
                'address' => $request->address,
                'role_id' => $request->role_id,
                'department_id' => $request->department_id,
                'branch_id' => $request->branch_id
            ];

            // Handle file uploads
            if ($request->hasFile('profile_pic')) {
                $profilePic = FileUpload::uploadImage($request->file('profile_pic'), 'profiles', 'profile');
                if (!$profilePic) {
                    return redirect()->back()->withInput()->with('error', 'Failed to upload profile picture.');
                }
                $data['profile_pic'] = $profilePic;
            }

            if ($request->hasFile('addhar_front_pic')) {
                $addharFront = FileUpload::uploadImage($request->file('addhar_front_pic'), 'documents', 'addhar_front');
                if (!$addharFront) {
                    return redirect()->back()->withInput()->with('error', 'Failed to upload Aadhar front picture.');
                }
                $data['addhar_front_pic'] = $addharFront;
            }

            if ($request->hasFile('addhar_back_pic')) {
                $addharBack = FileUpload::uploadImage($request->file('addhar_back_pic'), 'documents', 'addhar_back');
                if (!$addharBack) {
                    return redirect()->back()->withInput()->with('error', 'Failed to upload Aadhar back picture.');
                }
                $data['addhar_back_pic'] = $addharBack;
            }

            User::create($data);

            return redirect()->route('employee.index')->with('success', 'Employee created successfully');
        } catch (\Exception $e) {
            Log::error('Employee creation error: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to create employee. Please try again.');
        }
    }

    public function edit(User $employee)
    {
        try {
            $user = Auth::user();
            $userRole = Role::find($user->role_id);

            $states = State::whereNull('deleted_at')->where('status', 1)->get();
            $cities = City::whereNull('deleted_at')->where('status', 1)->get();
            $districts = District::whereNull('deleted_at')->get();

            // Filter roles based on user's authority level
            if ($userRole && $userRole->authorities) {
                $roles = Role::whereNull('deleted_at')
                    ->where('authorities', '>', $userRole->authorities)
                    ->get();
            } else {
                $roles = Role::whereNull('deleted_at')->get();
            }

            $departments = Department::whereNull('deleted_at')->where('status', 1)->get();
            
            // Filter branches based on user's role
            if ($user->role_id != 1) {
                // Non-admin users can only see their own branch
                $branches = Branch::whereNull('deleted_at')->where('status', 1)->where('id', $user->branch_id)->get();
            } else {
                $branches = Branch::whereNull('deleted_at')->where('status', 1)->get();
            }

            return view('team.edit-employee', compact('employee', 'states', 'cities', 'roles', 'districts', 'departments', 'branches'));
        } catch (\Exception $e) {
            Log::error('Error in EmployeeController@edit: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the edit form.');
        }
    }

    public function update(Request $request, User $employee)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'father_name' => 'nullable|string|max:255',
                'mother_name' => 'nullable|string|max:255',
                'dob' => 'nullable|date',
                'gender' => 'nullable|in:male,female,other',
                'marital_status' => 'nullable|in:single,married,divorced,widowed',
                'email' => 'required|email|unique:users,email,' . $employee->id,
                'mobile' => 'required|string|max:15|unique:users,mobile,' . $employee->id,
                'alternate_mobile' => 'nullable|string|max:15',
                'aadhaar_number' => 'nullable|string|size:12',
                'pan_number' => 'nullable|string|size:10',
                'joining_date' => 'required|date',
                'state_id' => 'required|exists:states,id',
                'district_id' => 'required|exists:districts,id',
                'city_id' => 'required|exists:cities,id',
                'pincode' => 'nullable|string|max:6',
                'address' => 'nullable|string',
                'role_id' => 'required|exists:roles,id',
                'department_id' => 'required|exists:departments,id',
                'branch_id' => 'required|exists:branches,id',
                'profile_pic' => 'nullable|image|mimes:jpeg,png,jpg|max:8048',
                'addhar_front_pic' => 'nullable|image|mimes:jpeg,png,jpg|max:8048',
                'addhar_back_pic' => 'nullable|image|mimes:jpeg,png,jpg|max:8048'
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $data = [
                'name' => $request->name,
                'father_name' => $request->father_name,
                'mother_name' => $request->mother_name,
                'dob' => $request->dob,
                'gender' => $request->gender,
                'marital_status' => $request->marital_status,
                'email' => $request->email,
                'mobile' => $request->mobile,
                'alternate_mobile' => $request->alternate_mobile,
                'aadhaar_number' => $request->aadhaar_number,
                'pan_number' => $request->pan_number,
                'joining_date' => $request->joining_date,
                'state_id' => $request->state_id,
                'district_id' => $request->district_id,
                'city_id' => $request->city_id,
                'pincode' => $request->pincode,
                'address' => $request->address,
                'role_id' => $request->role_id,
                'department_id' => $request->department_id,
                'branch_id' => $request->branch_id
            ];

            // Handle file uploads
            if ($request->hasFile('profile_pic')) {
                $profilePic = FileUpload::uploadImage($request->file('profile_pic'), 'profiles', 'profile');
                if (!$profilePic) {
                    return redirect()->back()->withInput()->with('error', 'Failed to upload profile picture.');
                }
                $data['profile_pic'] = $profilePic;
            }

            if ($request->hasFile('addhar_front_pic')) {
                $addharFront = FileUpload::uploadImage($request->file('addhar_front_pic'), 'documents', 'addhar_front');
                if (!$addharFront) {
                    return redirect()->back()->withInput()->with('error', 'Failed to upload Aadhar front picture.');
                }
                $data['addhar_front_pic'] = $addharFront;
            }

            if ($request->hasFile('addhar_back_pic')) {
                $addharBack = FileUpload::uploadImage($request->file('addhar_back_pic'), 'documents', 'addhar_back');
                if (!$addharBack) {
                    return redirect()->back()->withInput()->with('error', 'Failed to upload Aadhar back picture.');
                }
                $data['addhar_back_pic'] = $addharBack;
            }

            $employee->update($data);

            return redirect()->route('employee.index')->with('success', 'Employee updated successfully');
        } catch (\Exception $e) {
            Log::error('Employee update error: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to update employee. Please try again.');
        }
    }

    public function destroy(User $employee)
    {
        try {
            $employee->delete();
            return redirect()->back()->with('success', 'Employee deleted successfully');
        } catch (\Exception $e) {
            Log::error('Employee deletion error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete employee. Please try again.');
        }
    }

    public function toggleStatus(User $employee)
    {
        try {
            $employee->update(['status' => !$employee->status]);
            $status = $employee->status ? 'activated' : 'deactivated';
            return redirect()->back()->with('success', "Employee {$status} successfully");
        } catch (\Exception $e) {
            Log::error('Employee status toggle error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to update employee status. Please try again.');
        }
    }

    public function changePassword(Request $request, User $employee)
    {
        try {
            $validator = Validator::make($request->all(), [
                'new_password' => 'required|string|min:6',
                'confirm_password' => 'required|same:new_password'
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator);
            }

            $employee->update([
                'password' => Hash::make($request->new_password)
            ]);

            return redirect()->route('employee.index')->with('success', 'Password changed successfully');
        } catch (\Exception $e) {
            Log::error('Password change error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to change password. Please try again.');
        }
    }

    public function getCities($districtId)
    {
        try {
            $query = City::whereNull('deleted_at')
                ->where('status', 1);

            if ($districtId) {
                $query->where('district_id', $districtId);
            }

            return response()->json([
                'success' => true,
                'data' => $query->get()
            ]);
        } catch (\Exception $e) {
            Log::error('Error in EmployeeController@getCities: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while fetching cities.'], 500);
        }
    }

    public function getDistricts($stateId)
    {
        try {
            $query = District::whereNull('deleted_at')
                ->where('status', 1);

            if ($stateId) {
                $query->where('state_id', $stateId);
            }

            return response()->json([
                'success' => true,
                'data' => $query->get()
            ]);
        } catch (\Exception $e) {
            Log::error('Error in EmployeeController@getDistricts: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while fetching districts.'], 500);
        }
    }

    // Profile methods
    public function profileIndex()
    {
        try {
            $user = User::with(['branch', 'role', 'department', 'district', 'city', 'state'])->find(Auth::id());
            $states = State::whereNull('deleted_at')->where('status', 1)->get();
            $cities = City::whereNull('deleted_at')->where('status', 1)->get();
            $designations = collect(); // Empty collection since designations are not used in employee management
            $circles = collect(); // Empty collection since circles are not used in employee management

            return view('profile.profile-settings', compact('user', 'states', 'cities', 'designations', 'circles'));
        } catch (\Exception $e) {
            Log::error('Error in EmployeeController@profileIndex: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading profile.');
        }
    }

    public function profileUpdate(Request $request)
    {
        try {
            $user = Auth::user();

            $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email,' . $user->id,
                'mobile' => 'required|string|max:15|unique:users,mobile,' . $user->id,
                'state_id' => 'nullable|exists:states,id',
                'city_id' => 'nullable|exists:cities,id',
                'pincode' => 'nullable|string|max:10',
                'address' => 'nullable|string'
            ]);

            $user->update([
                'name' => $request->name,
                'email' => $request->email,
                'mobile' => $request->mobile,
                'state_id' => $request->state_id,
                'city_id' => $request->city_id,
                'pincode' => $request->pincode,
                'address' => $request->address
            ]);

            return redirect()->back()->with('success', 'Profile updated successfully');
        } catch (\Exception $e) {
            Log::error('Error in EmployeeController@profileUpdate: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating profile.');
        }
    }

    public function profileChangePassword(Request $request)
    {
        try {
            $request->validate([
                'current_password' => 'required',
                'new_password' => 'required|string|min:6',
                'confirm_password' => 'required|same:new_password'
            ]);

            $user = Auth::user();

            if (!Hash::check($request->current_password, $user->password)) {
                return redirect()->back()->with('error', 'Current password is incorrect');
            }

            $user->update([
                'password' => Hash::make($request->new_password)
            ]);

            return redirect()->back()->with('success', 'Password changed successfully');
        } catch (\Exception $e) {
            Log::error('Error in EmployeeController@profileChangePassword: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while changing password.');
        }
    }

    public function viewDocument($type, $filename)
    {
        try {
            $user = Auth::user();

            // Determine the correct path based on type
            if ($type === 'profile') {
                $path = public_path('storage/uploads/profiles/' . $filename);
            } else {
                $path = public_path('storage/uploads/documents/' . $filename);
            }

            if (!file_exists($path)) {
                abort(404, 'File not found');
            }

            return response()->file($path);
        } catch (\Exception $e) {
            Log::error('Error in EmployeeController@viewDocument: ' . $e->getMessage());
            abort(404, 'File not found');
        }
    }
}
