<?php

namespace App\Http\Controllers;

use App\Models\FormField;
use App\Models\FormFieldOption;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class FormFieldOptionController extends Controller
{
    public function index(FormField $formField)
    {
        try {
            $options = $formField->options()->with('parent')->orderBy('priority')->get();
            return view('admin.master.form-field-options.index', compact('formField', 'options'));
        } catch (\Exception $e) {
            Log::error('Error in FormFieldOptionController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching options.');
        }
    }

    public function create(FormField $formField)
    {
        try {
            $masterOptions = [];
            
            // If this field has a parent master field, get its options
            if ($formField->parent_field_id) {
                $masterField = FormField::find($formField->parent_field_id);
                if ($masterField) {
                    $masterOptions = $masterField->options()->whereNull('parent_id')->get();
                }
            }
            
            return view('admin.master.form-field-options.create', compact('formField', 'masterOptions'));
        } catch (\Exception $e) {
            Log::error('Error in FormFieldOptionController@create: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the create form.');
        }
    }

    public function store(Request $request, FormField $formField)
    {
        try {
            $validator = Validator::make($request->all(), [
                'option_value' => 'required|string|max:100',
                'parent_id' => 'nullable|exists:form_field_options,id',
                'priority' => 'integer|min:0',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $data = $request->only(['option_value', 'parent_id', 'priority', 'status']);
            
            $formField->options()->create($data);
            return redirect()->route('form-fields.options.index', $formField)->with('success', 'Option created successfully');
        } catch (\Exception $e) {
            Log::error('Error in FormFieldOptionController@store: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while creating the option.');
        }
    }

    public function edit(FormField $formField, FormFieldOption $option)
    {
        try {
            $masterOptions = [];
            
            // If this field has a parent master field, get its options
            if ($formField->parent_field_id) {
                $masterField = FormField::find($formField->parent_field_id);
                if ($masterField) {
                    $masterOptions = $masterField->options()->whereNull('parent_id')->get();
                }
            }
            
            return view('admin.master.form-field-options.edit', compact('formField', 'option', 'masterOptions'));
        } catch (\Exception $e) {
            Log::error('Error in FormFieldOptionController@edit: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the edit form.');
        }
    }

    public function update(Request $request, FormField $formField, FormFieldOption $option)
    {
        try {
            $validator = Validator::make($request->all(), [
                'option_value' => 'required|string|max:100',
                'parent_id' => 'nullable|exists:form_field_options,id',
                'priority' => 'integer|min:0',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $data = $request->only(['option_value', 'parent_id', 'priority', 'status']);
            
            $option->update($data);
            return redirect()->route('form-fields.options.index', $formField)->with('success', 'Option updated successfully');
        } catch (\Exception $e) {
            Log::error('Error in FormFieldOptionController@update: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating the option.');
        }
    }

    public function destroy(FormField $formField, FormFieldOption $option)
    {
        try {
            $option->delete();
            return redirect()->route('form-fields.options.index', $formField)->with('success', 'Option deleted successfully');
        } catch (\Exception $e) {
            Log::error('Error in FormFieldOptionController@destroy: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while deleting the option.');
        }
    }
}