<?php

namespace App\Http\Controllers;

use App\Models\Applicant;
use App\Models\CoApplicant;
use App\Models\ApplicantGuarantor;
use App\Models\ApplicantReference;
use App\Models\ApplicantLeadStatusLog;
use App\Models\FormField;
use App\Models\Branch;
use App\Models\User;
use App\Models\State;
use App\Models\City;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Utils\FileUpload;

class LeadController extends Controller
{
    public function index()
    {
        try {
            $applicants = Applicant::with(['creator', 'assignedTo'])
                ->orderBy('created_at', 'desc')
                ->paginate(10);

            return view('leads.index', compact('applicants'));
        } catch (\Exception $e) {
            Log::error('Error in LeadController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching leads.');
        }
    }

    public function create()
    {
        try {
            // $dropdownFields = FormField::with('options')->get()->keyBy('fie ld_name');
            $dropdownFields = FormField::with('options')->get()->keyBy('field_name');

            $branches = Branch::where('status', 1)->get();
            $users = User::where('status', 1)->where('role_id', 2)->get();
            $ro_users = User::where('status', 1)->where('role_id', 4)->get();
            $states = State::where('status', 1)->whereNull('deleted_at')->get();
            $cities = City::where('status', 1)->whereNull('deleted_at')->get();

            return view('admin.leads.create', compact('dropdownFields', 'branches', 'users', 'ro_users', 'states', 'cities'));
        } catch (\Exception $e) {
            Log::error('Error in LeadController@create: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the create form.');
        }
    }

    public function store(Request $request)
    {
        // dd($request->all());
        $request->validate([
            'applicant_first_name' => 'required|string|max:255',
            'applicant_last_name' => 'required|string|max:255',
            'father_name' => 'required|string|max:255',
            'contact_number' => 'required|string|max:20'
        ]);

        DB::beginTransaction();
        try {
            // Create Applicant
            $applicantData = array_filter($request->only([
                'segment',
                'loan_type',
                'customer_type',
                'applicant_first_name',
                'applicant_middle_name',
                'applicant_last_name',
                'father_name',
                'caste',
                'dob',
                'age',
                'gender',
                'contact_number',
                'contact_verified',
                'email',
                'address_type',
                'current_pincode',
                'current_state',
                'current_city',
                'current_address',
                'place_of_birth',
                'nationality',
                'marital_status',
                'spouse_name',
                'pan',
                'pan_verified',
                'aadhaar',
                'aadhaar_verified',
                'form_60',
                'voter_id',
                'voter_id_verified',
                'driving_license',
                'driving_license_verified',
                'passport',
                'banking_info',
                'login_date',
                'login_amount',
                'tenure',
                'roi',
                'branch_id',
                'bm_name_id',
                'rm_name',
                'bt_yes_no_id',
                'bt_company_name',
                'imd_waiver',
                'imd_transaction_ref',
                'business_address',
                'annual_income',
                'obligation',
                'foir',
                'ltv',
                'gps_tagging',
                'property_type',
                'property_owner_name',
                'property_address',
                'psl_yes_no_id',
                'manufacturing_service_agriculture',
                'psl_sub_category',
                'purpose_of_loans',
                'psl_document_type',
                'psl_document_number',
                'field_investigation_flag',
                'tele_verification_flag',
                'personal_discussion_flag',
                'status',
                'remark',
                'assign_to'
            ]), function ($value) {
                return $value !== null && $value !== '';
            });

            // Handle applicant image upload
            if ($request->hasFile('applicant_image')) {
                $applicantData['applicant_image'] = FileUpload::uploadImage($request->file('applicant_image'), 'applicants');
            }

            $applicant = Applicant::create(array_merge($applicantData, ['created_by' => Auth::id()]));

            // Log status change
            ApplicantLeadStatusLog::create([
                'applicant_lead_id' => $applicant->id,
                'lead_status' => $applicant->status ?? 1,
                'action_by' => Auth::id(),
                'remark' => $applicant->remark,
                'action_date' => now()
            ]);

            // Save References
            $referenceCount = 1;
            while ($request->has("reference_{$referenceCount}") && $request->input("reference_{$referenceCount}")) {
                ApplicantReference::create([
                    'applicant_id' => $applicant->id,
                    'name' => $request->input("reference_{$referenceCount}"),
                    'contact_no' => $request->input("reference_{$referenceCount}_contact_no"),
                    'relation_with_applicant' => $request->input("reference_{$referenceCount}_relation_id"),
                ]);
                $referenceCount++;
            }

            // Save Co-Applicants
            $coApplicantIndex = 1;
            while (true) {
                $prefix = $coApplicantIndex === 1 ? 'co_applicant_' : "co_applicant_{$coApplicantIndex}_";

                if (!$request->filled($prefix . 'first_name')) {
                    break;
                }

                $coApplicantData = array_filter([
                    'applicant_id' => $applicant->id,
                    'co_applicant_first_name' => $request->input($prefix . 'first_name'),
                    'applicant_middle_name' => $request->input($prefix . 'middle_name'),
                    'applicant_last_name' => $request->input($prefix . 'last_name'),
                    'father_name' => $request->input($prefix . 'father_name'),
                    'caste_id' => $request->input($prefix . 'caste') ?: null,
                    'dob' => $request->input($prefix . 'dob') ?: null,
                    'age' => $request->input($prefix . 'age') ?: null,
                    'gender_id' => $request->input($prefix . 'gender') ?: null,
                    'contact_number' => $request->input($prefix . 'contact_number'),
                    'address_type_id' => $request->input($prefix . 'address_type') ?: null,
                    'current_address' => $request->input($prefix . 'current_address'),
                    'current_city' => $request->input($prefix . 'current_city'),
                    'current_state' => $request->input($prefix . 'current_state'),
                    'current_pincode' => $request->input($prefix . 'current_pincode'),
                    'place_of_birth' => $request->input($prefix . 'place_of_birth'),
                    'nationality' => $request->input($prefix . 'nationality') ?: null,
                    'marital_status_id' => $request->input($prefix . 'marital_status') ?: null,
                    'spouse_name' => $request->input($prefix . 'spouse_name'),
                    'email' => $request->input($prefix . 'email'),
                    'pan' => $request->input($prefix . 'pan'),
                    'aadhaar' => $request->input($prefix . 'aadhaar'),
                    'voter_id' => $request->input($prefix . 'voter_id'),
                    'driving_license' => $request->input($prefix . 'driving_license'),
                    'passport' => $request->input($prefix . 'passport'),
                    'form_60' => $request->input($prefix . 'form_60') ?? 0,
                ], function ($value) {
                    return $value !== null && $value !== '';
                });

                // Handle co-applicant image upload
                $imageFieldName = $coApplicantIndex === 1 ? 'co_applicant_image' : "co_applicant_{$coApplicantIndex}_image";
                if ($request->hasFile($imageFieldName)) {
                    $coApplicantData['co_applicant_image'] = FileUpload::uploadImage($request->file($imageFieldName), 'co_applicants');
                }

                CoApplicant::create($coApplicantData);
                $coApplicantIndex++;
            }

            // Save Guarantor if provided
            if ($request->filled('guarantor_first_name')) {
                $guarantorData = array_filter([
                    'applicant_id' => $applicant->id,
                    'guarantor_first_name' => $request->guarantor_first_name,
                    'applicant_middle_name' => $request->guarantor_middle_name,
                    'applicant_last_name' => $request->guarantor_last_name,
                    'father_name' => $request->guarantor_father_name,
                    'caste_id' => $request->guarantor_caste ?: null,
                    'dob' => $request->guarantor_dob ?: null,
                    'age' => $request->guarantor_age ?: null,
                    'gender_id' => $request->guarantor_gender ?: null,
                    'contact_number' => $request->guarantor_contact_number,
                    'address_type_id' => $request->guarantor_address_type ?: null,
                    'current_address' => $request->guarantor_current_address,
                    'current_city' => $request->guarantor_current_city,
                    'current_state' => $request->guarantor_current_state,
                    'current_pincode' => $request->guarantor_current_pincode,
                    'place_of_birth' => $request->guarantor_place_of_birth,
                    'nationality' => $request->guarantor_nationality ?: null,
                    'marital_status_id' => $request->guarantor_marital_status ?: null,
                    'spouse_name' => $request->guarantor_spouse_name,
                    'email' => $request->guarantor_email,
                    'pan' => $request->guarantor_pan,
                    'aadhaar' => $request->guarantor_aadhaar,
                    'voter_id' => $request->guarantor_voter_id,
                    'driving_license' => $request->guarantor_driving_license,
                    'passport' => $request->guarantor_passport,
                    'form_60' => $request->guarantor_form_60 ?? 0,
                ], function ($value) {
                    return $value !== null && $value !== '';
                });

                // Handle guarantor image upload
                if ($request->hasFile('guarantor_image')) {
                    $guarantorData['applicant_guarantor_image'] = FileUpload::uploadImage($request->file('guarantor_image'), 'guarantors');
                }

                ApplicantGuarantor::create($guarantorData);
            }

            DB::commit();
            return redirect()->route('lead.create')->with('success', 'Lead created successfully.');
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error in LeadController@store: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Error creating lead: ' . $e->getMessage());
        }
    }

    public function show(Applicant $applicant)
    {
        try {
            $dropdownFields = FormField::with('options')->get()->keyBy('field_name');
            $applicant->load(['coApplicants', 'guarantors', 'references', 'creator', 'assignedTo', 'branch']);

            // Get status logs from database
            $statusLogs = ApplicantLeadStatusLog::where('applicant_lead_id', $applicant->id)
                ->with('actionBy')
                ->get()
                ->keyBy('lead_status');

            return view('admin.leads.show', compact('applicant', 'dropdownFields', 'statusLogs'));
        } catch (\Exception $e) {
            Log::error('Error in LeadController@show: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading lead details.');
        }
    }

    public function edit(Applicant $applicant)
    {
        try {
            $dropdownFields = FormField::with('options')->get()->keyBy('field_name');
            $states = State::all();
            $cities = City::all();
            $branches = Branch::all();
            $users = User::all();
            $ro_users = User::where('role_id', 4)->get();

            $applicant->load(['coApplicants', 'guarantors', 'references']);

            return view('admin.leads.edit', compact('applicant', 'dropdownFields', 'states', 'cities', 'branches', 'users', 'ro_users'));
        } catch (\Exception $e) {
            Log::error('Error in LeadController@edit: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading the edit form.');
        }
    }

    public function update(Request $request, Applicant $applicant)
    {
        try {
            $request->validate([
                'applicant_first_name' => 'required|string|max:255',
                'applicant_last_name' => 'required|string|max:255',
                'father_name' => 'required|string|max:255',
                'contact_number' => 'required|string|max:20',
                'current_address' => 'required|string',
                'current_pincode' => 'required|string|max:10',
                'place_of_birth' => 'required|string|max:255',
            ]);

            // Update applicant with all form data
            $applicantData = array_filter($request->only([
                'segment',
                'loan_type',
                'customer_type',
                'applicant_first_name',
                'applicant_middle_name',
                'applicant_last_name',
                'father_name',
                'caste',
                'dob',
                'age',
                'gender',
                'contact_number',
                'contact_verified',
                'email',
                'address_type',
                'current_pincode',
                'current_state',
                'current_city',
                'current_address',
                'place_of_birth',
                'nationality',
                'marital_status',
                'spouse_name',
                'pan',
                'pan_verified',
                'aadhaar',
                'aadhaar_verified',
                'form_60',
                'voter_id',
                'voter_id_verified',
                'driving_license',
                'driving_license_verified',
                'passport',
                'banking_info',
                'login_date',
                'login_amount',
                'tenure',
                'roi',
                'branch_id',
                'bm_name_id',
                'rm_name',
                'bt_yes_no_id',
                'bt_company_name',
                'imd_waiver',
                'imd_transaction_ref',
                'business_address',
                'annual_income',
                'obligation',
                'foir',
                'ltv',
                'gps_tagging',
                'property_type',
                'property_owner_name',
                'property_address',
                'psl_yes_no_id',
                'manufacturing_service_agriculture',
                'psl_sub_category',
                'purpose_of_loans',
                'psl_document_type',
                'psl_document_number',
                'field_investigation_flag',
                'tele_verification_flag',
                'personal_discussion_flag',
                'status',
                'assign_to'
            ]), function ($value) {
                return $value !== null && $value !== '';
            });

            $applicant->update($applicantData);

            return redirect()->route('lead.assign')
                ->with('success', 'Lead updated successfully.');
        } catch (\Exception $e) {
            Log::error('Error in LeadController@update: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating the lead.');
        }
    }

    public function destroy(Applicant $applicant)
    {
        DB::beginTransaction();
        try {
            // Soft delete related records
            CoApplicant::where('applicant_id', $applicant->id)->delete();
            ApplicantGuarantor::where('applicant_id', $applicant->id)->delete();
            ApplicantReference::where('applicant_id', $applicant->id)->delete();

            // Soft delete the applicant
            $applicant->delete();

            DB::commit();
            return redirect()->route('lead.assign')->with('success', 'Lead deleted successfully.');
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error in LeadController@destroy: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error deleting lead: ' . $e->getMessage());
        }
    }

    public function getCitiesByState($stateId)
    {
        try {
            $cities = City::where('state_id', $stateId)
                ->where('status', 1)
                ->whereNull('deleted_at')
                ->get(['id', 'name']);

            return response()->json($cities);
        } catch (\Exception $e) {
            Log::error('Error in LeadController@getCitiesByState: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while fetching cities.'], 500);
        }
    }

    public function assignLead(Request $request)
    {
        try {
            // dd($request->all());
            $query = Applicant::with(['creator', 'assignedTo', 'branch']);

            // Filter branches and users based on logged-in user's role
            $currentUser = Auth::user();
            if ($currentUser->role_id != 1) {
                // Non-admin users can only see leads from their own branch
                $query->where('branch_id', $currentUser->branch_id);
            }

            if ($request->filled('search')) {
                if (is_numeric($request->search)) {
                    $query->where('contact_number', 'like', '%' . $request->search . '%');
                } else {
                    $query->where(function ($q) use ($request) {
                        $q->where('applicant_first_name', 'like', '%' . $request->search . '%')
                            ->orWhere('applicant_last_name', 'like', '%' . $request->search . '%');
                    });
                }
            }

            if ($request->filled('status')) {
                if (is_array($request->status)) {
                    $query->whereIn('status', $request->status);
                } else {
                    $query->where('status', $request->status);
                }
            }

            if ($request->filled('created_by')) {
                if (is_array($request->created_by)) {
                    $query->whereIn('created_by', $request->created_by);
                } else {
                    $query->where('created_by', $request->created_by);
                }
            }

            if ($request->filled('branch_id')) {
                if (is_array($request->branch_id)) {
                    $query->whereIn('branch_id', $request->branch_id);
                } else {
                    $query->where('branch_id', $request->branch_id);
                }
            }

            if ($request->filled('assign_to')) {
                if (is_array($request->assign_to)) {
                    $query->whereIn('assign_to', $request->assign_to);
                } else {
                    $query->where('assign_to', $request->assign_to);
                }
            }

            if ($request->filled('start_date') && $request->filled('end_date')) {
                $query->whereBetween('created_at', [$request->start_date . ' 00:00:00', $request->end_date . ' 23:59:59']);
            } elseif ($request->filled('start_date')) {
                $query->where('created_at', '>=', $request->start_date . ' 00:00:00');
            } elseif ($request->filled('end_date')) {
                $query->where('created_at', '<=', $request->end_date . ' 23:59:59');
            }

            $applicants = $query->orderBy('created_at', 'desc')->paginate(10);
            $applicants->appends($request->query());

            $branches = Branch::where('status', 1)->get();
            $users = User::where('status', 1)->get();
            $bmUsers = User::where('status', 1)->where('role_id', 2)->get();
            $roUsers = User::where('status', 1)->where('role_id', 4)->get();

            if ($currentUser->role_id != 1) {
                // Non-admin users can only see their own branch
                $branches = Branch::where('status', 1)->where('id', $currentUser->branch_id)->get();
                $roUsers = User::where('status', 1)->where('role_id', 4)->where('branch_id', $currentUser->branch_id)->get();
            }
            return view('admin.leads.assign', compact('applicants', 'branches', 'users', 'bmUsers', 'roUsers', 'currentUser'));
        } catch (\Exception $e) {
            Log::error('Error in LeadController@assignLead: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while loading lead assignments.');
        }
    }

    public function getBranchStaff($branchId)
    {
        try {
            $staff = User::where('status', 1)
                ->where('role_id', 4)
                ->where('branch_id', $branchId)
                ->get(['id', 'name']);

            return response()->json($staff);
        } catch (\Exception $e) {
            Log::error('Error in LeadController@getBranchStaff: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while fetching branch staff.'], 500);
        }
    }

    public function branchVerify(Applicant $applicant)
    {
        try {
            $applicant->update(['status' => 4]);

            // Log status change
            ApplicantLeadStatusLog::create([
                'applicant_lead_id' => $applicant->id,
                'lead_status' => 4,
                'remark' =>  auth()->user()->name . ' is verifying this lead ',
                'action_by' => Auth::id(),
                'action_date' => now()
            ]);

            $dropdownFields = FormField::with('options')->get()->keyBy('field_name');
            $applicant->load(['coApplicants', 'guarantors', 'references', 'creator', 'assignedTo', 'branch']);

            // Get status logs from database
            $statusLogs = ApplicantLeadStatusLog::where('applicant_lead_id', $applicant->id)
                ->with('actionBy')
                ->get()
                ->keyBy('lead_status');

            return view('admin.leads.show', compact('applicant', 'dropdownFields', 'statusLogs'));
        } catch (\Exception $e) {
            Log::error('Error in LeadController@branchVerify: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error updating lead status');
        }
    }

    public function updateAssignment(Request $request)
    {
        $request->validate([
            'applicant_id' => 'required|exists:applicants,id',
            'branch_id' => 'required|exists:branches,id',
            'assign_to' => 'required|exists:users,id'
        ]);

        try {
            $applicant = Applicant::findOrFail($request->applicant_id);
            $applicant->update([
                'assign_to' => $request->assign_to,
                'assign_date' => now(),
                'status' => 2,
                'assigned_by' => Auth::id(),
                'branch_id' => $request->branch_id
            ]);

            // Log status change
            ApplicantLeadStatusLog::create([
                'applicant_lead_id' => $applicant->id,
                'lead_status' => 2,
                'remark' => 'Lead assigned to ' . User::find($request->assign_to)->name . ' please review',
                'action_by' => Auth::id(),
                'action_date' => now()
            ]);

            return response()->json(['success' => true, 'message' => 'Lead assigned successfully']);
        } catch (\Exception $e) {
            Log::error('Error in LeadController@updateAssignment: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => $e->getMessage()]);
        }
    }
}
