<?php

namespace App\Http\Controllers;

use App\Models\Menu;
use App\Models\SubMenu;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class MenuController extends Controller
{
    public function index()
    {
        try {
            $menus = Menu::whereNull('deleted_at')->orderBy('priority')->get();
            return view('masters.menu.menu', compact('menus'));
        } catch (\Exception $e) {
            Log::error('Error in MenuController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching menus.');
        }
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'menu_name' => 'required|string|max:255',
                'priority' => 'required|integer|min:0'
            ]);

            Menu::create([
                'name' => $request->menu_name,
                'priority' => $request->priority,
                'status' => 1
            ]);
            
            return redirect()->back()->with('success', 'Menu created successfully');
        } catch (\Exception $e) {
            Log::error('Error in MenuController@store: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while creating the menu.');
        }
    }

    public function update(Request $request, Menu $menu)
    {
        try {
            $request->validate([
                'menu_name' => 'required|string|max:255',
                'priority' => 'required|integer|min:0'
            ]);

            $menu->update([
                'name' => $request->menu_name,
                'priority' => $request->priority
            ]);

            return redirect()->back()->with('success', 'Menu updated successfully');
        } catch (\Exception $e) {
            Log::error('Error in MenuController@update: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating the menu.');
        }
    }

    public function destroy(Menu $menu)
    {
        try {
            DB::transaction(function () use ($menu) {
                $menu->delete();
            });
            return redirect()->back()->with('success', 'Menu deleted successfully');
        } catch (\Exception $e) {
            Log::error('Error in MenuController@destroy: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while deleting the menu.');
        }
    }

    public function toggleStatus(Menu $menu)
    {
        try {
            $menu->update(['status' => !$menu->status]);
            $status = $menu->status ? 'activated' : 'deactivated';
            return redirect()->back()->with('success', "Menu {$status} successfully");
        } catch (\Exception $e) {
            Log::error('Error in MenuController@toggleStatus: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating menu status.');
        }
    }

    public function allSubMenus()
    {
        try {
            $menus = Menu::whereNull('deleted_at')->orderBy('priority')->get();
            $subMenus = SubMenu::with('menu')
                ->whereNull('deleted_at')
                ->orderBy('priority')
                ->get();
            
            return view('masters.menu.all-submenu', compact('menus', 'subMenus'));
        } catch (\Exception $e) {
            Log::error('Error in MenuController@allSubMenus: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching submenus.');
        }
    }

    public function subMenus($menuId)
    {
        try {
            $menu = Menu::findOrFail($menuId);
            $subMenus = SubMenu::where('menu_id', $menuId)
                ->whereNull('deleted_at')
                ->orderBy('priority')
                ->get();
            
            return view('masters.menu.submenu', compact('menu', 'subMenus'));
        } catch (\Exception $e) {
            Log::error('Error in MenuController@subMenus: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching submenus.');
        }
    }

    public function storeSubMenu(Request $request, $menuId)
    {
        try {
            $request->validate([
                'submenu_name' => 'required|string|max:255',
                'parent_id' => 'nullable|exists:sub_menus,id',
                'route' => 'nullable|string|max:255',
                'priority' => 'required|integer|min:0'
            ]);

            SubMenu::create([
                'menu_id' => $menuId,
                'parent_id' => $request->parent_id,
                'name' => $request->submenu_name,
                'route' => $request->route,
                'priority' => $request->priority,
                'status' => 1
            ]);
            
            return redirect()->back()->with('success', 'Submenu created successfully');
        } catch (\Exception $e) {
            Log::error('Error in MenuController@storeSubMenu: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while creating the submenu.');
        }
    }

    public function updateSubMenu(Request $request, $menuId, SubMenu $submenu)
    {
        try {
            $request->validate([
                'submenu_name' => 'required|string|max:255',
                'parent_id' => 'nullable|exists:sub_menus,id',
                'route' => 'nullable|string|max:255',
                'priority' => 'required|integer|min:0'
            ]);

            $submenu->update([
                'parent_id' => $request->parent_id,
                'name' => $request->submenu_name,
                'route' => $request->route,
                'priority' => $request->priority
            ]);

            return redirect()->back()->with('success', 'Submenu updated successfully');
        } catch (\Exception $e) {
            Log::error('Error in MenuController@updateSubMenu: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating the submenu.');
        }
    }

    public function destroySubMenu($menuId, SubMenu $submenu)
    {
        try {
            DB::transaction(function () use ($submenu) {
                $submenu->delete();
            });
            return redirect()->back()->with('success', 'Submenu deleted successfully');
        } catch (\Exception $e) {
            Log::error('Error in MenuController@destroySubMenu: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while deleting the submenu.');
        }
    }

    public function toggleSubMenuStatus($menuId, SubMenu $submenu)
    {
        try {
            $submenu->update(['status' => !$submenu->status]);
            $status = $submenu->status ? 'activated' : 'deactivated';
            return redirect()->back()->with('success', "Submenu {$status} successfully");
        } catch (\Exception $e) {
            Log::error('Error in MenuController@toggleSubMenuStatus: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while updating submenu status.');
        }
    }
}