<?php

namespace App\Http\Controllers;

use App\Models\State;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class StateController extends Controller
{
    public function index()
    {
        try {
            $states = State::whereNull('deleted_at')->paginate(10);
            return view('masters.location.state', compact('states'));
        } catch (\Exception $e) {
            Log::error('Error in StateController@index: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while fetching states.');
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'state_name' => 'required|string|max:255|unique:states,name,NULL,id,deleted_at,NULL'
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            State::create([
                'name' => $request->state_name
            ]);
            
            return redirect()->back()->with('success', 'State created successfully');
        } catch (\Exception $e) {
            Log::error('State creation error: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to create state. Please try again.');
        }
    }

    public function update(Request $request, State $state)
    {
        try {
            $validator = Validator::make($request->all(), [
                'state_name' => 'required|string|max:255|unique:states,name,' . $state->id . ',id,deleted_at,NULL'
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $state->update([
                'name' => $request->state_name
            ]);

            return redirect()->back()->with('success', 'State updated successfully');
        } catch (\Exception $e) {
            Log::error('State update error: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to update state. Please try again.');
        }
    }

    public function destroy(State $state)
    {
        try {
            $state->delete();
            return redirect()->back()->with('success', 'State deleted successfully');
        } catch (\Exception $e) {
            Log::error('State deletion error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete state. Please try again.');
        }
    }

    public function toggleStatus(State $state)
    {
        try {
            $state->update(['status' => !$state->status]);
            $status = $state->status ? 'activated' : 'deactivated';
            return redirect()->back()->with('success', "State {$status} successfully");
        } catch (\Exception $e) {
            Log::error('State status toggle error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to update state status. Please try again.');
        }
    }
}