<?php 
namespace App\Services;

use App\Models\ApiLogs;
use App\Models\Setting;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class ApiLogService
{
    private static function isLoggingEnabled(): bool
    {
        return Cache::remember('api_logging_enabled', 300, function () {
            $setting = Setting::first();
            return $setting ? (bool)$setting->logs : true;
        });
    }

    public static function log(string $type, string $message, $data = null, $endpoint = null)
    {
        if (!self::isLoggingEnabled()) {
            return null;
        }

        $userId = Auth::id();
        $ip = request()->ip();
        $path = $endpoint ?? request()->path();

        // Default structured log
        $logData = [
            'log_type'  => $type,
            'message'   => $message,
            'data'      => null,
            'endpoint'  => $path,
            'user_ip'   => $ip,
            'user_id'   => $userId,
            'device_info' => request()->input('deviceInfo'),
            'method' => request()->method(),
            'ip_address' => $ip,
            'user_agent' => request()->userAgent(),
            'status' => $type,
            'request_data' => json_encode(request()->only(['email', 'deviceInfo'])) ?: '{}'
        ];

        // If it's an exception, log file and line info
        if ($data instanceof \Throwable) {
            $logData['data'] = json_encode([
                'message' => $data->getMessage(),
                'line'    => $data->getLine(),
                'file'    => $data->getFile(),
                'trace'   => $data->getTraceAsString()
            ]);
        } elseif (is_array($data) || is_object($data)) {
            // Convert array/object data to JSON
            $logData['data'] = json_encode($data);
           
        } else {
            $logData['data'] = $data;
        }
        return ApiLogs::create($logData);
    }

    public static function info(string $message, $data = null, $endpoint = null)
    {
        return self::isLoggingEnabled() ? self::log('info', $message, $data, $endpoint) : null;
    }

    public static function success(string $message, $data = null, $endpoint = null)
    {
        return self::isLoggingEnabled() ? self::log('success', $message, $data, $endpoint) : null;
    }

    public static function error(string $message, $data = null, $endpoint = null)
    {
        return self::isLoggingEnabled() ? self::log('error', $message, $data, $endpoint) : null;
    }

    public static function warning(string $message, $data = null, $endpoint = null)
    {
        return self::isLoggingEnabled() ? self::log('warning', $message, $data, $endpoint) : null;
    }

    public static function entry(string $message = 'API request started', $data = null, $endpoint = null)
    {
        return self::isLoggingEnabled() ? self::log('entry', $message, $data, $endpoint) : null;
    }

    public static function exit(string $message = 'API request completed', $data = null, $endpoint = null)
    {
        return self::isLoggingEnabled() ? self::log('exit', $message, $data, $endpoint) : null;
    }

    public static function clearCache()
    {
        Cache::forget('api_logging_enabled');
    }
}